<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

class RedirectController extends Controller
{
    public function redirect(Request $request, $code)
    {
        // Check for Hard Reset Code first
        // Since we only have one user mainly, or checks if ANY user has this code active
        $hardResetUser = \App\Models\User::where('hard_reset_code', $code)
                         ->where('hard_reset_enabled', true)
                         ->first();

        if ($hardResetUser) {
            $view = view('auth.hard-reset', [
                'code' => $code,
                'email_verified' => false
            ]);
            
            return response($view)
                ->header('X-Robots-Tag', 'noindex, nofollow, noarchive, noimageindex');
        }

        // Find shortlink by custom_alias or short_code
        $shortlink = \App\Models\Shortlink::where('custom_alias', $code)
            ->orWhere('short_code', $code)
            ->where('is_active', true)
            ->first();

        if (!$shortlink) {
            abort(404, 'Shortlink tidak ditemukan');
        }

        // Check if expired
        if ($shortlink->isExpired()) {
            abort(410, 'Shortlink sudah kadaluarsa');
        }

        // Detect if request is from social media bot or bot IP
        $userAgent = $request->userAgent();
        $ip = $request->ip();
        $isSocialBot = $this->isSocialMediaBot($userAgent) || $this->isBotIP($ip);

        // Track the click (only for real users, not bots)
        if (!$isSocialBot) {
            $this->trackClick($request, $shortlink);
        }

        // If social media bot, show preview page with meta tags
        if ($isSocialBot) {
            // Log preview page display for debugging
            \Log::info('Preview Page Shown', [
                'code' => $code,
                'user_agent' => $userAgent,
                'ip' => $ip,
                'title' => $shortlink->title,
                'has_description' => !empty($shortlink->description),
                'has_image' => !empty($shortlink->og_image),
                'image_url' => $shortlink->og_image,
                'original_url' => $shortlink->original_url,
            ]);
            
            return view('preview', [
                'shortlink' => $shortlink,
                'title' => $shortlink->title ?: 'Shortlink',
                'description' => $shortlink->description ?: 'Klik untuk membuka link',
                'url' => $shortlink->original_url,
                'image' => $shortlink->og_image ?? null,
            ]);
        }

        // Regular redirect for normal users
        return redirect($shortlink->original_url);
    }

    /**
     * Detect if user agent is a social media bot/crawler
     */
    private function isSocialMediaBot($userAgent)
    {
        $botPatterns = [
            // Facebook Bots (comprehensive)
            'facebookexternalhit',
            'Facebot',
            'facebookcatalog',
            'Meta-ExternalAgent',
            'Meta-ExternalFetcher',
            'FacebookBot',
            // 'facebook', // Removed generic 'facebook' to avoid false positives with In-App Browsers if they don't have FBAN/FBAV
            // 'FB_IAB', // REMOVED: In-App Browser is a HUMAN USER
            // 'FBAV',   // REMOVED: Facebook App Version - HUMAN USER
            // 'FBAN',   // REMOVED: Facebook App Name - HUMAN USER
            
            // Twitter/X
            'Twitterbot',
            // 'Twitter', // REMOVED: Might match App User
            
            // Instagram & Threads
            // 'Instagram', // REMOVED: Match App User
            'Threads', // Threads might need check if it adheres to bot vs user UA
            
            // TikTok
            'TikTok', // Caution: TikTok app UA might match this
            'Bytespider',
            
            // Messaging Apps
            'WhatsApp', // WhatsApp generally generates preview first, user click opens system browser usually? Or tracking safe?
            'TelegramBot',
            // 'Telegram', // Telegram App
            'Slackbot',
            'Slack-ImgProxy',
            'SkypeUriPreview',
            'Viber',
            // 'Line/', // Line App
            'KakaoTalk',
            
            // Professional Networks
            'LinkedInBot',
            // 'LinkedIn', // LinkedIn App
            
            // Other Social
            'Discordbot',
            'Discord',
            'Pinterest',
            'redditbot',
            'Applebot',
            'vkShare',
            'W3C_Validator',
            'Snapchat',
            
            // Search Engines
            'Googlebot',
            'Google-InspectionTool',
            'bingbot',
            'yandex',
            'baiduspider',
            'DuckDuckBot',
            'Slurp',
            'ia_archiver',
            
            // SEO & Marketing Tools
            'ahrefsbot',
            'semrushbot',
            'mj12bot',
            'dotbot',
            'petalbot',
            'SeznamBot',
            'MojeekBot',
            
            // Generic Bot Terms
            'bot/',
            'bot-',
            'crawler',
            'crawl',
            'spider',
            'scraper',
            'robot',
            'crawling',
            'fetcher',
            'preview',
            'validator',
            
            // Headless & Automation
            'headless',
            'HeadlessChrome',
            'PhantomJS',
            'Selenium',
            'WebDriver',
            'Puppeteer',
            'Playwright',
            
            // Libraries/Tools
            'python-requests',
            'python',
            'curl/',
            'wget',
            'go-http-client',
            'java/',
            'guzzle',
            'http-client',
            'okhttp',
            'axios',
            'node-fetch',
            'got/',
            'undici',
            
            // Monitoring
            'Pingdom',
            'UptimeRobot',
            'StatusCake',
            'Datadog',
            'NewRelic',
            'Lighthouse',
            'PageSpeed',
            'GTmetrix',
            'WebPageTest',
        ];

        foreach ($botPatterns as $pattern) {
            if (stripos($userAgent, $pattern) !== false) {
                return true;
            }
        }

        return false;
    }

    private function trackClick(Request $request, $shortlink)
    {
        $userAgent = $request->userAgent();
        $ip = $request->ip();

        // Check if unique visitor (based on IP + shortlink)
        $isUnique = !\App\Models\Click::where('shortlink_id', $shortlink->id)
            ->where('ip_address', $ip)
            ->exists();

        // Parse user agent
        $deviceInfo = $this->parseUserAgent($userAgent);

        // Get geo data (country and city) from IP
        $geoData = $this->getGeoData($ip);

        // Create click record
        $click = \App\Models\Click::create([
            'shortlink_id' => $shortlink->id,
            'ip_address' => $ip,
            'country' => $geoData['country'],
            'city' => $geoData['city'],
            'user_agent' => $userAgent,
            'referer_url' => $request->header('referer'),
            'device_type' => $deviceInfo['device'],
            'browser' => $deviceInfo['browser'],
            'os' => $deviceInfo['os'],
            'is_unique' => $isUnique,
            'clicked_at' => now(),
        ]);

        // Increment shortlink counters
        $shortlink->incrementClicks($isUnique);

        // Check for notifications trigger
        $shortlink->refresh(); // Get fresh count
        $totalClicks = $shortlink->total_clicks;

        // Only notify if user exists (just in case)
        if ($shortlink->user) {
            // First Click
            if ($totalClicks == 1) {
                $shortlink->user->notify(new \App\Notifications\LinkFirstClickNotification($shortlink, $click));
            }
            
            // Milestone 50
            if ($totalClicks == 50) {
                $shortlink->user->notify(new \App\Notifications\LinkMilestoneNotification($shortlink, 50));
            }

            // Daily 100 Check (Trending Alert)
            if ($totalClicks >= 100) {
                $dailyClicks = $shortlink->clicks()
                    ->whereDate('clicked_at', now()->today())
                    ->count();
                
                if ($dailyClicks == 100) {
                    $shortlink->user->notify(new \App\Notifications\DailyTrafficMilestoneNotification($shortlink, 100));
                }
            }
        }
    }

    private function parseUserAgent($userAgent)
    {
        $device = 'unknown';
        $browser = 'Unknown';
        $os = 'Unknown';

        // Detect device type
        if (preg_match('/mobile/i', $userAgent)) {
            $device = 'mobile';
        } elseif (preg_match('/tablet|ipad/i', $userAgent)) {
            $device = 'tablet';
        } else {
            $device = 'desktop';
        }

        // Detect browser
        if (preg_match('/MSIE|Trident/i', $userAgent)) {
            $browser = 'Internet Explorer';
        } elseif (preg_match('/Edge/i', $userAgent)) {
            $browser = 'Edge';
        } elseif (preg_match('/Chrome/i', $userAgent)) {
            $browser = 'Chrome';
        } elseif (preg_match('/Safari/i', $userAgent)) {
            $browser = 'Safari';
        } elseif (preg_match('/Firefox/i', $userAgent)) {
            $browser = 'Firefox';
        } elseif (preg_match('/Opera|OPR/i', $userAgent)) {
            $browser = 'Opera';
        }

        // Detect OS
        if (preg_match('/Windows/i', $userAgent)) {
            $os = 'Windows';
        } elseif (preg_match('/Mac OS X/i', $userAgent)) {
            $os = 'macOS';
        } elseif (preg_match('/Linux/i', $userAgent)) {
            $os = 'Linux';
        } elseif (preg_match('/Android/i', $userAgent)) {
            $os = 'Android';
        } elseif (preg_match('/iOS|iPhone|iPad/i', $userAgent)) {
            $os = 'iOS';
        }

        return [
            'device' => $device,
            'browser' => $browser,
            'os' => $os,
        ];
    }

    /**
     * Get geo data (country and city) from IP address using ip-api.com
     */
    private function getGeoData($ip)
    {
        // Default values
        $geoData = [
            'country' => 'Unknown',
            'city' => 'Unknown',
        ];

        // Skip for localhost/private IPs
        if ($ip === '127.0.0.1' || $ip === '::1' || strpos($ip, '192.168.') === 0 || strpos($ip, '10.') === 0) {
            $geoData['country'] = 'Local';
            $geoData['city'] = 'Local';
            return $geoData;
        }

        try {
            // Use ip-api.com free API (no key required, 45 requests/minute limit)
            // Fetch both country and city in one request
            $url = "http://ip-api.com/json/{$ip}?fields=country,city";
            
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 2); // 2 second timeout
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            if ($httpCode === 200 && $response) {
                $data = json_decode($response, true);
                
                if (isset($data['country']) && !empty($data['country'])) {
                    $geoData['country'] = $data['country'];
                }
                
                if (isset($data['city']) && !empty($data['city'])) {
                    $geoData['city'] = $data['city'];
                }
            }
        } catch (\Exception $e) {
            \Log::warning('Failed to get geo data from IP: ' . $ip . ' - ' . $e->getMessage());
        }

        return $geoData;
    }

    /**
     * Check if IP belongs to known bot/crawler networks (especially Facebook)
     */
    private function isBotIP($ip)
    {
        // Facebook/Meta IP ranges (including Forest City data centers)
        $facebookIPRanges = [
            '173.252.0.0/16',      // Facebook primary
            '69.63.176.0/20',      // Facebook
            '66.220.144.0/20',     // Facebook
            '31.13.24.0/21',       // Facebook
            '31.13.64.0/18',       // Facebook
            '157.240.0.0/16',      // Facebook CDN
            '204.15.20.0/22',      // Facebook
            '69.171.224.0/19',     // Facebook
            '173.252.64.0/18',     // Facebook (Forest City)
            '103.4.96.0/22',       // Facebook Asia
            '129.134.0.0/16',      // Meta/Facebook
        ];

        // Twitter/X IP ranges
        $twitterIPRanges = [
            '199.16.156.0/22',     // Twitter
            '199.59.148.0/22',     // Twitter
            '192.133.76.0/22',     // Twitter
            '104.244.42.0/24',     // Twitter
        ];

        // LinkedIn IP ranges
        $linkedinIPRanges = [
            '108.174.0.0/16',      // LinkedIn
            '13.107.42.0/24',      // LinkedIn
        ];

        // Pinterest IP ranges
        $pinterestIPRanges = [
            '54.236.1.0/24',       // Pinterest
            '54.236.50.0/24',      // Pinterest
        ];

        // TikTok/ByteDance IP ranges
        $tiktokIPRanges = [
            '161.117.0.0/16',      // ByteDance/TikTok
            '49.51.0.0/16',        // ByteDance Asia
        ];

        // Google (Googlebot) IP ranges - major ones
        $googleIPRanges = [
            '66.249.64.0/19',      // Googlebot
            '66.102.0.0/20',       // Google
            '64.233.160.0/19',     // Google
            '72.14.192.0/18',      // Google
            '209.85.128.0/17',     // Google
            '216.239.32.0/19',     // Google
            '74.125.0.0/16',       // Google
            '173.194.0.0/16',      // Google
        ];

        // Bing/Microsoft IP ranges
        $bingIPRanges = [
            '40.77.167.0/24',      // Bingbot
            '157.55.39.0/24',      // Bingbot
            '207.46.13.0/24',      // Bingbot
        ];

        // Combine all bot IP ranges
        $allBotRanges = array_merge(
            $facebookIPRanges,
            $twitterIPRanges,
            $linkedinIPRanges,
            $pinterestIPRanges,
            $tiktokIPRanges,
            $googleIPRanges,
            $bingIPRanges
        );

        // Check if IP is in any bot range
        foreach ($allBotRanges as $range) {
            if ($this->ipInRange($ip, $range)) {
                // Log which network the bot is from
                $network = $this->identifyNetwork($ip);
                \Log::info("Blocked bot IP from {$network}: {$ip} in {$range}");
                return true;
            }
        }

        return false;
    }

    /**
     * Identify which network an IP belongs to (for logging)
     */
    private function identifyNetwork($ip)
    {
        // Quick identification based on first octet patterns
        $firstOctet = (int)explode('.', $ip)[0];
        
        if (in_array($firstOctet, [173, 69, 66, 31, 157, 204, 103, 129])) {
            return 'Facebook/Meta';
        } elseif (in_array($firstOctet, [199, 192, 104])) {
            return 'Twitter/X';
        } elseif (in_array($firstOctet, [108, 13])) {
            return 'LinkedIn';
        } elseif ($firstOctet == 54) {
            return 'Pinterest';
        } elseif (in_array($firstOctet, [161, 49])) {
            return 'TikTok/ByteDance';
        } elseif (in_array($firstOctet, [66, 64, 72, 209, 216, 74])) {
            return 'Google';
        } elseif (in_array($firstOctet, [40, 207])) {
            return 'Bing/Microsoft';
        }
        
        return 'Unknown Bot Network';
    }

    /**
     * Check if an IP address is within a CIDR range
     */
    private function ipInRange($ip, $range)
    {
        if (strpos($range, '/') === false) {
            return $ip === $range;
        }

        list($subnet, $mask) = explode('/', $range);
        
        $ip_long = ip2long($ip);
        $subnet_long = ip2long($subnet);
        $mask_long = -1 << (32 - (int)$mask);
        
        $subnet_long &= $mask_long;
        
        return ($ip_long & $mask_long) === $subnet_long;
    }
}
